<?php
declare(strict_types=1);

namespace GHL_CRM\Tests\Core\Stubs {

class LoaderSingletonComponent
{
    public static int $initCalls = 0;
    public static int $instances = 0;

    private static ?self $instance = null;

    public static function get_instance(): self
    {
        if (null === self::$instance) {
            self::$instance = new self();
            self::$instances++;
        }

        return self::$instance;
    }

    public function init(): void
    {
        self::$initCalls++;
    }

    public static function reset(): void
    {
        self::$instance  = null;
        self::$initCalls = 0;
        self::$instances = 0;
    }
}

class LoaderStandardComponent
{
    public static int $initCalls = 0;
    public static int $instances = 0;

    public function __construct()
    {
        self::$instances++;
    }

    public function init(): void
    {
        self::$initCalls++;
    }

    public static function reset(): void
    {
        self::$initCalls = 0;
        self::$instances = 0;
    }
}

}

namespace GHL_CRM\Tests\Core {

use Brain\Monkey;
use Brain\Monkey\Functions;
use GHL_CRM\Core\Loader;
use GHL_CRM\Tests\Core\Stubs\LoaderSingletonComponent;
use GHL_CRM\Tests\Core\Stubs\LoaderStandardComponent;
use PHPUnit\Framework\TestCase;

class LoaderTest extends TestCase
{
    protected function setUp(): void
    {
        parent::setUp();
        Monkey\setUp();

        if (! defined('GHL_CRM_PATH')) {
            define('GHL_CRM_PATH', __DIR__ . '/../');
        }

        if (! defined('GHL_CRM_BASENAME')) {
            define('GHL_CRM_BASENAME', 'ghl-crm-integration/plugin.php');
        }

        if (! defined('MINUTE_IN_SECONDS')) {
            define('MINUTE_IN_SECONDS', 60);
        }

        Functions\when('register_activation_hook')->justReturn();
        Functions\when('register_deactivation_hook')->justReturn();
        Functions\when('add_action')->justReturn();
        Functions\when('add_filter')->justReturn();
        Functions\when('do_action')->justReturn();

        Functions\when('apply_filters')->alias(static fn($hook, $value, ...$args) => $value);

        LoaderSingletonComponent::reset();
        LoaderStandardComponent::reset();
    }

    protected function tearDown(): void
    {
        $this->resetLoaderSingleton();

        Monkey\tearDown();
        parent::tearDown();
    }

    public function testInitComponentsInitializesFilteredComponentsOnce(): void
    {
        $components = array(
            'singleton' => LoaderSingletonComponent::class,
            'standard'  => LoaderStandardComponent::class,
        );

        Functions\when('apply_filters')->alias(static function ($hook, $value, ...$args) use ($components) {
            return 'ghl_crm_loader_components' === $hook ? $components : $value;
        });

        $loader = Loader::get_instance();
        $loader->init_components();

        $this->assertSame(1, LoaderSingletonComponent::$instances);
        $this->assertSame(1, LoaderSingletonComponent::$initCalls);
        $this->assertSame(1, LoaderStandardComponent::$instances);
        $this->assertSame(1, LoaderStandardComponent::$initCalls);

        $loader->get_component('singleton');
        $loader->get_component('standard');

        $this->assertSame(1, LoaderSingletonComponent::$instances);
        $this->assertSame(1, LoaderSingletonComponent::$initCalls);
        $this->assertSame(1, LoaderStandardComponent::$instances);
        $this->assertSame(1, LoaderStandardComponent::$initCalls);
    }

    public function testGetComponentReturnsNullWhenUnknown(): void
    {
        Functions\when('apply_filters')->alias(static function ($hook, $value, ...$args) {
            return 'ghl_crm_loader_components' === $hook ? array() : $value;
        });

    $loader = Loader::get_instance();

    $this->assertNull($loader->get_component('missing'));
    }

    private function resetLoaderSingleton(): void
    {
        $reflection = new \ReflectionClass(Loader::class);
        $instance   = $reflection->getProperty('instance');
        $instance->setAccessible(true);
        $instance->setValue(null, null);
    }
}

}
