<?php
declare(strict_types=1);

namespace {
    if ( ! class_exists( 'WooCommerce' ) ) {
        class WooCommerce {
        }
    }
}

namespace GHL_CRM\Tests\Sync {

use Brain\Monkey;
use Brain\Monkey\Functions;
use GHL_CRM\Sync\ContactCache;
use GHL_CRM\Sync\QueueProcessor;
use GHL_CRM\Sync\RateLimiter;
use PHPUnit\Framework\TestCase;

class QueueProcessorTest extends TestCase
{
    protected function setUp(): void
    {
        parent::setUp();
        Monkey\setUp();

        Functions\when( 'absint' )->alias( 'intval' );
    Functions\when( 'esc_html' )->alias( static fn( $value ) => $value );
    Functions\when( 'esc_html__' )->alias( static fn( $value ) => $value );
    Functions\when( '__' )->alias( static fn( $value ) => $value );
        Functions\when( 'do_action' )->justReturn();
    }

    protected function tearDown(): void
    {
        Monkey\tearDown();
        $this->resetQueueProcessorSingleton();
        parent::tearDown();
    }

    public function testExecuteSyncInvokesRegisteredHandler(): void
    {
        $processor = $this->makeProcessor();

        $captured = null;
        $processor->register_handler(
            'user',
            function (string $action, int $itemId, array $payload) use (&$captured) {
                $captured = [ $action, $itemId, $payload ];
                return [ 'success' => true ];
            }
        );

        $result = $processor->execute_sync('user', 'do_something', 42, [ 'foo' => 'bar' ]);

        $this->assertSame([ 'success' => true ], $result);
        $this->assertSame([ 'do_something', 42, [ 'foo' => 'bar' ] ], $captured);
        $this->assertTrue($processor->has_handler('user'));
    }

    public function testExecuteSyncDelegatesToFiltersWhenHandlerMissing(): void
    {
        $processor = $this->makeProcessor();

        $payload = [ 'foo' => 'bar' ];

        Functions\expect('apply_filters')
            ->once()
            ->with('ghl_crm_execute_order_sync', false, 'sync', 7, $payload)
            ->andReturn([ 'handled' => true ]);

        $result = $processor->execute_sync('ORDER', 'sync', 7, $payload);

        $this->assertSame([ 'handled' => true ], $result);
    }

    public function testDefaultHandlersRegistered(): void
    {
        $processor = $this->makeProcessor();

        $this->assertTrue( $processor->has_handler( 'user' ) );
        $this->assertTrue( $processor->has_handler( 'contact' ) );
        $this->assertTrue( $processor->has_handler( 'wc_customer' ) );
    }

    public function testRegisterHandlerIsCaseInsensitive(): void
    {
        $processor = $this->makeProcessor();
        $processor->register_handler('CustomType', static function (): bool {
            return true;
        });

        $this->assertTrue($processor->has_handler('customtype'));
        $this->assertTrue($processor->has_handler('CUSTOMTYPE'));
    }

    public function testDependencyWaitsWhenIncomplete(): void
    {
        $events    = [];
        $processor = $this->makeProcessor(function (string $event, array $context) use (&$events): void {
            $events[] = $event;
        });

        $handlerCalled = false;
        $processor->register_handler('custom', function () use (&$handlerCalled): bool {
            $handlerCalled = true;
            return true;
        });

        $mocked_wpdb = new class {
            public string $prefix = 'wp_';
            public function prepare($query, $id)
            {
                return sprintf($query, $id);
            }
            public function get_var($query)
            {
                return 'processing';
            }
        };

        global $wpdb;
        $previous_wpdb = $wpdb ?? null;
        $wpdb          = $mocked_wpdb;

        try {
            $result = $processor->execute_sync('custom', 'run', 55, [ '_depends_on_queue_id' => 99 ]);
        } finally {
            $wpdb = $previous_wpdb;
        }

        $this->assertFalse($handlerCalled);
        $this->assertSame(
            [
                'success' => false,
                'error'   => 'Waiting for dependency to complete',
                'skip'    => true,
            ],
            $result
        );
        $this->assertContains('dependency_detected', $events);
        $this->assertContains('dependency_waiting', $events);
    }

    public function testDependencyAllowsExecutionWhenComplete(): void
    {
        $events    = [];
        $processor = $this->makeProcessor(function (string $event, array $context) use (&$events): void {
            $events[] = $event;
        });

        $processor->register_handler('custom', static function (): array {
            return [ 'ok' => true ];
        });

        $mocked_wpdb = new class {
            public string $prefix = 'wp_';
            public function prepare($query, $id)
            {
                return sprintf($query, $id);
            }
            public function get_var($query)
            {
                return 'completed';
            }
        };

        global $wpdb;
        $previous_wpdb = $wpdb ?? null;
        $wpdb          = $mocked_wpdb;

        try {
            $result = $processor->execute_sync('custom', 'run', 42, [ '_depends_on_queue_id' => 100 ]);
        } finally {
            $wpdb = $previous_wpdb;
        }

        $this->assertSame([ 'ok' => true ], $result);
        $this->assertContains('dependency_ready', $events);
        $this->assertContains('after_execute', $events);
    }

    public function testExecuteViaFiltersDefaultFallback(): void
    {
        $processor = $this->makeProcessor();

        $payload = [ 'payload' => true ];

        Functions\expect('apply_filters')
            ->once()
            ->with('ghl_crm_execute_sync', false, 'unknown', 'act', 5, $payload)
            ->andReturn([ 'delegated' => true ]);

        $result = $processor->execute_sync('unknown', 'act', 5, $payload);

        $this->assertSame([ 'delegated' => true ], $result);
    }

    public function testExecuteViaOrderFilter(): void
    {
        $processor = $this->makeProcessor();

        $payload = [ 'order' => 12 ];

        Functions\expect('apply_filters')
            ->once()
            ->with('ghl_crm_execute_order_sync', false, 'process', 12, $payload)
            ->andReturn([ 'order_handled' => true ]);

        $result = $processor->execute_sync('ORDER', 'process', 12, $payload);

        $this->assertSame([ 'order_handled' => true ], $result);
    }

    public function testWooCommerceHandlerUsesFactory(): void
    {
        $wc_stub = new class {
            public array $calls = [];
            public function process_customer_conversion(array $payload)
            {
                $this->calls[] = ['convert_lead', $payload];
                return ['converted' => true];
            }
            public function process_product_tags(array $payload)
            {
                $this->calls[] = ['apply_tags', $payload];
                return ['tags' => true];
            }
            public function process_opportunity_sync(array $payload)
            {
                $this->calls[] = ['opportunity', $payload];
                return ['opportunity' => true];
            }
        };

        $factory = static function () use ($wc_stub) {
            return $wc_stub;
        };

        $processor = $this->makeProcessor(null, $factory);

        $result = $processor->execute_sync('wc_customer', 'convert_lead', 10, ['id' => 1]);
        $this->assertSame(['converted' => true], $result);

        $processor->execute_sync('wc_customer', 'apply_tags', 10, ['id' => 1]);
    $processor->execute_sync('wc_customer', 'create_opportunity', 10, ['id' => 1]);

    $this->assertCount(3, $wc_stub->calls);
    $this->assertSame('convert_lead', $wc_stub->calls[0][0]);
    $this->assertSame('apply_tags', $wc_stub->calls[1][0]);
    $this->assertSame('opportunity', $wc_stub->calls[2][0]);
    }

    public function testErrorEventDispatchedWhenHandlerThrows(): void
    {
        $events = [];
        $processor = $this->makeProcessor(function (string $event, array $context) use (&$events): void {
            $events[] = $event;
        });

        $processor->register_handler('user', static function (): void {
            throw new \RuntimeException('Handler failure');
        });

        try {
            $processor->execute_sync('user', 'run', 99, []);
            $this->fail('Expected exception was not thrown.');
        } catch (\RuntimeException $exception) {
            $this->assertSame('Handler failure', $exception->getMessage());
        }

        $this->assertContains('before_execute', $events);
        $this->assertContains('error', $events);
    }

    private function makeProcessor(
        ?callable $eventDispatcher = null,
        ?callable $woocommerceFactory = null,
        ?callable $clientFactory = null,
        ?callable $contactResourceFactory = null
    ): QueueProcessor
    {
        $rateLimiter = $this->getMockBuilder(RateLimiter::class)
            ->disableOriginalConstructor()
            ->getMock();

        $contactCache = $this->getMockBuilder(ContactCache::class)
            ->disableOriginalConstructor()
            ->getMock();

        if (null === $eventDispatcher) {
            $eventDispatcher = static function (): void {
                // Intentionally empty for tests.
            };
        }

        return QueueProcessor::get_instance(
            $rateLimiter,
            $contactCache,
            $eventDispatcher,
            $woocommerceFactory,
            $clientFactory,
            $contactResourceFactory
        );
    }

    private function resetQueueProcessorSingleton(): void
    {
    $reflection = new \ReflectionClass(QueueProcessor::class);
    $instance   = $reflection->getProperty('instance');
    $instance->setAccessible(true);
    $instance->setValue(null, null);
    }
}

}
